package com.fireflye.proyectos.rest;

import java.util.List;

import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.fireflye.proyectos.dto.Proyectos;
import com.fireflye.proyectos.service.ProyectosException;
import com.fireflye.proyectos.service.ProyectosService;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import lombok.extern.slf4j.Slf4j;

@RestController
@RequestMapping("/servicios/proyectos")
@Slf4j
public class ProyectosRest {

	@Autowired
	private ProyectosService proyectosService;

	@PostMapping(value = "/")
	@Operation(summary = "Agregar proyecto")
	@ApiResponses(value = { @ApiResponse(responseCode = "202", description = "Proyecto agregado"),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<Void> addProyecto(@RequestBody @Valid Proyectos proyectos) {
		try {
			log.info("addProyecto", proyectos);
			proyectosService.addProyecto(proyectos);
			return ResponseEntity.accepted().body(null);
		} catch (ProyectosException e) {
			log.error("Exception", e);
			return ResponseEntity.badRequest().body(null);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@Operation(summary = "Actualización de proyectos")
	@ApiResponses(value = { @ApiResponse(responseCode = "202", description = "Proyecto actualizado"),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	@PutMapping(value = "/")
	public ResponseEntity<Void> updateProyecto(@RequestBody @Valid Proyectos proyectos) {
		try {
			log.info("updateProyecto", proyectos);
			proyectosService.updateProyecto(proyectos);
			return ResponseEntity.accepted().body(null);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@GetMapping(value = "/")
	@Operation(summary = "Listado de proyecto")
	@ApiResponses(value = { @ApiResponse(responseCode = "200", description = "Listado de proyectos"),
			@ApiResponse(responseCode = "204", description = "No hay usuarios registrados en el sistema", content = @Content),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<List<Proyectos>> listProyectos() {
		try {
			log.info("listProyectos");
			List<Proyectos> resultado = proyectosService.listProyectos();

			if (resultado == null || resultado.isEmpty()) {
				return ResponseEntity.noContent().build();
			}

			return ResponseEntity.ok().body(resultado);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@GetMapping(value = "/{idProyecto}")
	@Operation(summary = "Consulta de proyecto")
	@ApiResponses(value = { @ApiResponse(responseCode = "200", description = "Proyecto consultado"),
			@ApiResponse(responseCode = "404", description = "No se encuentra el usuario en el sistema", content = @Content),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<Proyectos> getProyecto(@PathVariable Integer idProyecto) {
		try {
			log.info("getProyecto", idProyecto);
			Proyectos resultado = proyectosService.getProyecto(idProyecto);

			if (resultado == null) {
				return ResponseEntity.notFound().build();
			}

			return ResponseEntity.ok().body(resultado);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@DeleteMapping(value = "/{idProyecto}")
	@Operation(summary = "Eliminación de proyectos")
	@ApiResponses(value = { @ApiResponse(responseCode = "202", description = "Proyecto eliminado"),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<Void> deleteProyecto(@PathVariable Integer idProyecto) {
		try {
			log.info("deleteProyecto", idProyecto);
			proyectosService.deleteProyecto(idProyecto);

			return ResponseEntity.accepted().body(null);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

}
