package com.fireflye.usuarios.rest;

import java.util.List;

import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.fireflye.usuarios.dto.Usuarios;
import com.fireflye.usuarios.service.UsuariosException;
import com.fireflye.usuarios.service.UsuariosService;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import lombok.extern.slf4j.Slf4j;

@RestController
@RequestMapping("/servicios/usuarios")
@Slf4j
public class UsuariosRest {

	@Autowired
	private UsuariosService usuariosService;

	@PostMapping(value = "/")
	@Operation(summary = "Agregar usuarios")
	@ApiResponses(value = { @ApiResponse(responseCode = "202", description = "Usuario agregado"),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<Void> addUsuario(@RequestBody @Valid Usuarios usuarios) {
		try {
			log.info("addUsuario", usuarios);
			usuariosService.addUsuario(usuarios);
			return ResponseEntity.accepted().body(null);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@GetMapping(value = "/validate/{idUsuario}")
	@Operation(summary = "Validación de usuarios")
	@ApiResponses(value = { @ApiResponse(responseCode = "200", description = "Usuario validado correctamente"),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<Void> validateUsuario(@PathVariable Integer idUsuario) {
		try {
			log.info("addUsuario", idUsuario);
			usuariosService.validateUsuario(idUsuario);
			return ResponseEntity.ok().body(null);
		} catch (UsuariosException e) {
			log.error("Exception", e);
			return ResponseEntity.badRequest().body(null);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@PutMapping(value = "/")
	@Operation(summary = "Actualización de usuarios")
	@ApiResponses(value = { @ApiResponse(responseCode = "202", description = "Usuario actualizado"),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<Void> updateUsuario(@RequestBody @Valid Usuarios usuarios) {
		try {
			log.info("updateUsuario", usuarios);
			usuariosService.updateUsuario(usuarios);
			return ResponseEntity.accepted().body(null);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@GetMapping(value = "/")
	@Operation(summary = "Listado de usuarios")
	@ApiResponses(value = { @ApiResponse(responseCode = "200", description = "Listado de usuarios"),
			@ApiResponse(responseCode = "204", description = "No hay usuarios registrados en el sistema", content = @Content),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<List<Usuarios>> listUsuarios() {
		try {
			log.info("listUsuarios");
			List<Usuarios> resultado = usuariosService.listUsuarios();

			if (resultado == null || resultado.isEmpty()) {
				return ResponseEntity.noContent().build();
			}

			return ResponseEntity.ok().body(resultado);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@GetMapping(value = "/{idUsuario}")
	@Operation(summary = "Consulta de usuario")
	@ApiResponses(value = { @ApiResponse(responseCode = "200", description = "Usuario consultado"),
			@ApiResponse(responseCode = "404", description = "No se encuentra el usuario en el sistema", content = @Content),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<Usuarios> getUsuario(@PathVariable Integer idUsuario) {
		try {
			log.info("getUsuario", idUsuario);
			Usuarios resultado = usuariosService.getUsuario(idUsuario);

			if (resultado == null) {
				return ResponseEntity.notFound().build();
			}

			return ResponseEntity.ok().body(resultado);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

	@DeleteMapping(value = "/{idUsuario}")
	@Operation(summary = "Eliminación de usuarios")
	@ApiResponses(value = { @ApiResponse(responseCode = "202", description = "Usuario eliminado"),
			@ApiResponse(responseCode = "400", description = "Datos recibidos inválidos", content = @Content),
			@ApiResponse(responseCode = "500", description = "Error en procesamiento de información", content = @Content) })
	public ResponseEntity<Void> deleteUsuario(@PathVariable Integer idUsuario) {
		try {
			log.info("deleteUsuario", idUsuario);
			usuariosService.deleteUsuario(idUsuario);

			return ResponseEntity.accepted().body(null);
		} catch (Exception e) {
			log.error("Exception", e);
			return ResponseEntity.internalServerError().body(null);
		}
	}

}
